// popup.js - Authecrim PIN-based authentication

const BACKEND_URL = window.CAPYNEWS_CONFIG?.BACKEND_URL || 'http://localhost:3000';
const EXTENSION_VERSION = '2.6.0'; // Must match version in manifest.json

// PIN state
let currentPin = '';
let currentEmail = '';

// Initialize when DOM elements are ready (called by popup-loader.js)
window.initPopup = async function() {
  // DOM Elements
  window.loginView = document.getElementById('loginView');
  window.loggedInView = document.getElementById('loggedInView');
  window.loadingView = document.getElementById('loading');
  window.errorDiv = document.getElementById('error');
  window.emailInput = document.getElementById('email');
  window.requestPinBtn = document.getElementById('requestPinBtn');
  window.pinSection = document.getElementById('pinSection');
  window.pinDisplay = document.getElementById('pinDisplay');
  window.pinPad = document.getElementById('pinPad');
  window.verifyPinBtn = document.getElementById('verifyPinBtn');
  window.logoutBtn = document.getElementById('logoutBtn');
  window.userName = document.getElementById('userName');
  window.userEmail = document.getElementById('userEmail');

  setupPinPad();
  setupEventListeners();
  await checkAuthState();
};

// Setup PIN pad handlers
function setupPinPad() {
  const pinButtons = window.pinPad.querySelectorAll('.pin-btn');
  pinButtons.forEach(btn => {
    btn.addEventListener('click', () => {
      const digit = btn.dataset.digit;
      if (digit === 'clear') {
        clearPin();
      } else if (digit === 'backspace') {
        if (currentPin.length > 0) {
          currentPin = currentPin.slice(0, -1);
          updatePinDisplay();
        }
      } else if (currentPin.length < 6) {
        addPinDigit(digit);
      }
    });
  });

  // Add keyboard support for PIN entry
  document.addEventListener('keydown', (e) => {
    // Only handle keys when PIN section is visible
    if (window.pinSection.style.display !== 'block') return;

    if (e.key >= '0' && e.key <= '9') {
      e.preventDefault();
      if (currentPin.length < 6) {
        addPinDigit(e.key);
      }
    } else if (e.key === 'Backspace' || e.key === 'Delete') {
      e.preventDefault();
      if (currentPin.length > 0) {
        currentPin = currentPin.slice(0, -1);
        updatePinDisplay();
      }
    } else if (e.key === 'Escape') {
      e.preventDefault();
      clearPin();
    } else if (e.key === 'Enter' && currentPin.length === 6) {
      e.preventDefault();
      window.verifyPinBtn.click();
    }
  });
}

function addPinDigit(digit) {
  if (currentPin.length < 6) {
    currentPin += digit;
    updatePinDisplay();
  }
}

function clearPin() {
  currentPin = '';
  updatePinDisplay();
}

function updatePinDisplay() {
  const digits = window.pinDisplay.querySelectorAll('.pin-digit');
  digits.forEach((digit, index) => {
    if (index < currentPin.length) {
      digit.textContent = currentPin[index];
      digit.classList.add('filled');
    } else {
      digit.textContent = '_';
      digit.classList.remove('filled');
    }
  });

  // Enable/disable verify button based on PIN length
  if (window.verifyPinBtn) {
    window.verifyPinBtn.disabled = currentPin.length !== 6;
  }

  // Auto-submit when 6 digits are entered
  if (currentPin.length === 6) {
    setTimeout(() => {
      if (window.verifyPinBtn && !window.verifyPinBtn.disabled) {
        window.verifyPinBtn.click();
      }
    }, 300); // Small delay for visual feedback
  }
}

// Setup event listeners
function setupEventListeners() {
  // Request PIN
  window.requestPinBtn.addEventListener('click', requestPin);

  // Verify PIN
  window.verifyPinBtn.addEventListener('click', verifyPin);

  // Allow Enter key on email
  window.emailInput.addEventListener('keypress', (e) => {
    if (e.key === 'Enter' && !window.requestPinBtn.disabled) {
      requestPin();
    }
  });

  // Request PIN again button
  const requestPinAgainBtn = document.getElementById('requestPinAgainBtn');
  if (requestPinAgainBtn) {
    requestPinAgainBtn.addEventListener('click', () => {
      if (currentEmail) {
        requestPin();
      }
    });
  }

  // Logout
  window.logoutBtn.addEventListener('click', logout);
}

// Check if user is already logged in
async function checkAuthState() {
  showLoading(true);
  hideError();

  try {
    const result = await chrome.storage.local.get(['authToken', 'user', 'pendingEmail']);

    if (result.authToken && result.user) {
      const response = await fetch(`${BACKEND_URL}/api/me`, {
        headers: {
          'Authorization': `Bearer ${result.authToken}`,
          'X-Extension-Version': EXTENSION_VERSION
        }
      });

      if (response.status === 426) {
        const data = await response.json();
        showUpdateRequired(data);
        return;
      }

      if (response.ok) {
        showLoggedIn(result.user);
      } else {
        await chrome.storage.local.remove(['authToken', 'user']);
        // Check if there's a pending PIN request
        if (result.pendingEmail) {
          showLoginWithPin(result.pendingEmail);
        } else {
          showLogin();
        }
      }
    } else if (result.pendingEmail) {
      // Show PIN pad if PIN was requested
      showLoginWithPin(result.pendingEmail);
    } else {
      showLogin();
    }
  } catch (error) {
    console.error('Auth check failed:', error);
    showLogin();
  }

  showLoading(false);
}

// Request PIN
async function requestPin() {
  const email = window.emailInput.value.trim().toLowerCase();

  if (!email || !email.includes('@')) {
    showError('Please enter a valid email address');
    return;
  }

  if (!email.endsWith('@cloudwalk.io')) {
    showError('Only @cloudwalk.io emails are allowed');
    return;
  }

  showLoading(true);
  hideError();
  window.requestPinBtn.disabled = true;

  try {
    const response = await fetch(`${BACKEND_URL}/api/auth/authecrim/request`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'X-Extension-Version': EXTENSION_VERSION
      },
      body: JSON.stringify({ email })
    });

    const data = await response.json();

    if (response.status === 426) {
      showUpdateRequired(data);
      showLoading(false);
      window.requestPinBtn.disabled = false;
      return;
    }

    if (!response.ok) {
      throw new Error(data.error || 'Failed to request PIN');
    }

    // Show PIN pad IMMEDIATELY (before loading finishes)
    currentEmail = email;
    await chrome.storage.local.set({ pendingEmail: email });

    // Ensure login view is visible and PIN pad is shown
    showLoading(false);
    window.loginView.style.display = 'block';
    window.loginView.classList.add('pin-active'); // Hide email section, show PIN pad
    window.pinSection.style.display = 'block';
    clearPin();
    window.requestPinBtn.textContent = 'PIN Requested';

    // Show info message
    showError('PIN sent to Slack! Enter the 6-digit code below:', 'info');

  } catch (error) {
    showError(error.message || 'Failed to request PIN');
  }

  // Only hide loading if PIN pad wasn't shown (error case)
  if (window.pinSection.style.display !== 'block') {
    showLoading(false);
  }
  window.requestPinBtn.disabled = false;
}

// Verify PIN
async function verifyPin() {
  if (currentPin.length !== 6) {
    showError('Please enter 6-digit PIN');
    return;
  }

  showLoading(true);
  hideError();
  window.verifyPinBtn.disabled = true;

  try {
    const response = await fetch(`${BACKEND_URL}/api/auth/authecrim/verify`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'X-Extension-Version': EXTENSION_VERSION
      },
      body: JSON.stringify({
        email: currentEmail,
        pin: currentPin
      })
    });

    const data = await response.json();

    if (response.status === 426) {
      showUpdateRequired(data);
      showLoading(false);
      window.verifyPinBtn.disabled = false;
      return;
    }

    if (!response.ok) {
      throw new Error(data.error || 'Invalid PIN');
    }

    // Clear PIN from memory immediately (security!)
    currentPin = '';
    currentEmail = '';
    clearPin();

    // Save token and clear pending email
    await chrome.storage.local.set({
      authToken: data.token,
      user: data.user
    });
    await chrome.storage.local.remove(['pendingEmail']);

    showLoggedIn(data.user);

  } catch (error) {
    showError(error.message || 'PIN verification failed');
    clearPin(); // Clear on error too
  }

  showLoading(false);
  window.verifyPinBtn.disabled = false;
}

// Logout
async function logout() {
  // Clear PIN state
  currentPin = '';
  currentEmail = '';
  clearPin();

  await chrome.storage.local.remove(['authToken', 'user', 'pendingEmail']);
  window.loginView.classList.remove('pin-active'); // Show email section again
  window.pinSection.style.display = 'none';
  window.requestPinBtn.textContent = 'Send PIN to Slack';
  showLogin();
}

// UI Helpers
function showLoading(show) {
  window.loadingView.style.display = show ? 'block' : 'none';
  if (show) {
    window.loginView.style.display = 'none';
    window.loggedInView.style.display = 'none';
  }
}

function showLogin() {
  window.loadingView.style.display = 'none';
  window.loginView.style.display = 'block';
  window.loggedInView.style.display = 'none';
  window.loginView.classList.remove('pin-active'); // Show email section again
  window.pinSection.style.display = 'none';
  currentPin = '';
  currentEmail = '';
  window.emailInput.value = '';
  clearPin();
  window.requestPinBtn.textContent = 'Send PIN to Slack';
}

function showLoginWithPin(email) {
  window.loadingView.style.display = 'none';
  window.loginView.style.display = 'block';
  window.loggedInView.style.display = 'none';
  currentEmail = email;
  window.emailInput.value = email;
  window.loginView.classList.add('pin-active'); // Hide email section, show PIN pad
  window.pinSection.style.display = 'block';
  window.requestPinBtn.textContent = 'PIN Requested';
  clearPin();
  showError('PIN sent to Slack! Enter the 6-digit code below:', 'info');
}

function showLoggedIn(user) {
  window.loadingView.style.display = 'none';
  window.loginView.style.display = 'none';
  window.loggedInView.style.display = 'block';

  window.userName.textContent = user.name || 'CloudWalk User';
  window.userEmail.textContent = user.email || 'Authenticated';
}

function showError(message, type = 'error') {
  window.errorDiv.textContent = message;
  window.errorDiv.style.display = 'block';
  if (type === 'info') {
    window.errorDiv.style.background = 'rgba(0, 217, 255, 0.1)';
    window.errorDiv.style.borderColor = 'rgba(0, 217, 255, 0.3)';
    window.errorDiv.style.color = '#00d9ff';
  } else {
    window.errorDiv.style.background = 'rgba(255, 100, 100, 0.1)';
    window.errorDiv.style.borderColor = 'rgba(255, 100, 100, 0.3)';
    window.errorDiv.style.color = '#ff6464';
  }
}

function hideError() {
  window.errorDiv.style.display = 'none';
}

function showUpdateRequired(data) {
  window.loadingView.style.display = 'none';
  window.loginView.style.display = 'block';
  window.loggedInView.style.display = 'none';

  const message = `UPDATE REQUIRED: Extension v${data.currentVersion} is outdated. Minimum required: v${data.minVersion}`;
  window.errorDiv.innerHTML = `
    ${message}<br>
    <a href="${data.updateUrl}" target="_blank" style="color: #0ff; text-decoration: underline; cursor: pointer;">
      [ CLICK TO UPDATE ]
    </a>
  `;
  window.errorDiv.style.display = 'block';
  window.errorDiv.style.background = 'rgba(255, 165, 0, 0.1)';
  window.errorDiv.style.borderColor = 'rgba(255, 165, 0, 0.5)';
  window.errorDiv.style.color = '#ffa500';
}
