// background.js - Service worker for CapyNews extension
console.log('[CapyNews] Background service worker loaded');

// Import config
importScripts('config.js');

const BACKEND_URL = self.CAPYNEWS_CONFIG?.BACKEND_URL || 'http://localhost:3000';
const EXTENSION_VERSION = '2.2.0'; // Must match version in manifest.json

// Function to inject content script manually if needed
async function ensureContentScriptInjected(tabId) {
  try {
    await chrome.tabs.sendMessage(tabId, { action: "ping" });
    return true;
  } catch (error) {
    try {
      await chrome.scripting.executeScript({
        target: { tabId: tabId },
        files: ['content.js']
      });
      return true;
    } catch (err) {
      console.error('[CapyNews] Failed to inject content script:', err);
      return false;
    }
  }
}

// Create context menu on install
chrome.runtime.onInstalled.addListener(() => {
  console.log('[CapyNews] Extension installed');
  chrome.contextMenus.create({
    id: "analyzeWithCapyNews",
    title: "Analyze with CapyNews 🦫",
    contexts: ["page", "selection"] // Works on whole page or selected text
  });
});

// Handle messages from content script
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  console.log('[CapyNews] Received message:', request.action);

  if (request.action === "analyzeContent") {
    handleAnalyzeRequest(request)
      .then(result => sendResponse(result))
      .catch(error => sendResponse({ success: false, error: error.message }));
    return true; // Keep message channel open for async response
  }

  if (request.action === "checkAuth") {
    checkAuthStatus()
      .then(result => sendResponse(result))
      .catch(error => sendResponse({ authenticated: false, error: error.message }));
    return true;
  }

  if (request.action === "fetchChannels") {
    chrome.storage.local.get(['authToken']).then(result => {
      if (!result.authToken) {
        sendResponse({ success: false, error: 'Not authenticated' });
      } else {
        fetchChannels(result.authToken)
          .then(data => sendResponse(data || { success: false, error: 'Failed to fetch channels' }))
          .catch(error => sendResponse({ success: false, error: error.message }));
      }
    });
    return true;
  }
});

// Check if user is authenticated
async function checkAuthStatus() {
  const result = await chrome.storage.local.get(['authToken', 'user']);
  
  if (!result.authToken) {
    return { authenticated: false };
  }
  
  // Verify token is still valid
  try {
    const response = await fetch(`${BACKEND_URL}/api/me`, {
      headers: {
        'Authorization': `Bearer ${result.authToken}`,
        'X-Extension-Version': EXTENSION_VERSION
      }
    });

    if (response.status === 426) {
      const data = await response.json();
      return {
        authenticated: false,
        updateRequired: true,
        updateInfo: data
      };
    }

    if (response.ok) {
      return { authenticated: true, user: result.user };
    } else {
      // Token expired, clear it
      await chrome.storage.local.remove(['authToken', 'user']);
      return { authenticated: false };
    }
  } catch (error) {
    console.error('[CapyNews] Auth check failed:', error);
    return { authenticated: false, error: 'Network error' };
  }
}

// Fetch available channels
async function fetchChannels(authToken) {
  try {
    const response = await fetch(`${BACKEND_URL}/api/channels`, {
      headers: {
        'Authorization': `Bearer ${authToken}`,
        'X-Extension-Version': EXTENSION_VERSION
      }
    });

    if (!response.ok) {
      console.error('[CapyNews] Failed to fetch channels');
      return null;
    }

    const data = await response.json();
    return data;
  } catch (error) {
    console.error('[CapyNews] Error fetching channels:', error);
    return null;
  }
}

// Handle analyze request - send to backend
async function handleAnalyzeRequest(request) {
  const { url, selectedText, fullContent, channelId } = request;

  // Get auth token
  const result = await chrome.storage.local.get(['authToken']);

  if (!result.authToken) {
    return {
      success: false,
      error: 'Not authenticated. Please sign in via the extension popup.',
      needsAuth: true
    };
  }

  console.log('[CapyNews] Sending to backend:', url, 'Channel:', channelId || 'default');

  try {
    const response = await fetch(`${BACKEND_URL}/api/analyze`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${result.authToken}`,
        'X-Extension-Version': EXTENSION_VERSION
      },
      body: JSON.stringify({
        url,
        selectedText,
        fullContent,
        channelId: channelId || 'default'
      })
    });

    const data = await response.json();

    if (!response.ok) {
      // Check if update is required
      if (response.status === 426) {
        return {
          success: false,
          error: `Update required: ${data.message}`,
          updateRequired: true,
          updateInfo: data
        };
      }

      // Check if it's an auth error
      if (response.status === 401) {
        await chrome.storage.local.remove(['authToken', 'user']);
        return {
          success: false,
          error: 'Session expired. Please sign in again.',
          needsAuth: true
        };
      }
      throw new Error(data.error || 'Failed to analyze content');
    }
    
    return { 
      success: true, 
      data: data.data,
      slackPosted: data.slackPosted
    };
    
  } catch (error) {
    console.error('[CapyNews] Backend error:', error);
    return { success: false, error: error.message };
  }
}

// Context menu click handler
chrome.contextMenus.onClicked.addListener(async (info, tab) => {
  console.log('[CapyNews] Context menu clicked for tab:', tab?.id);
  
  if (info.menuItemId === "analyzeWithCapyNews" && tab?.id) {
    try {
      // First check auth
      const authStatus = await checkAuthStatus();
      
      if (!authStatus.authenticated) {
        // Open popup to prompt login
        chrome.action.openPopup();
        return;
      }
      
      const tabInfo = await chrome.tabs.get(tab.id);
      
      if (tabInfo.status === 'complete') {
        const isInjected = await ensureContentScriptInjected(tab.id);
        
        if (!isInjected) {
          console.error('[CapyNews] Could not inject content script');
          return;
        }
        
        // Small delay to ensure content script is ready
        setTimeout(async () => {
          try {
            const response = await chrome.tabs.sendMessage(tab.id, { action: "getSelection" });
            console.log('[CapyNews] Content script response:', response);
          } catch (error) {
            console.error('[CapyNews] Message error:', error);
          }
        }, 100);
      }
    } catch (error) {
      console.error('[CapyNews] Error:', error);
    }
  }
});
